function [x, singular] = fpGelim(A,b)
% x = fpGelim(A,b) - Solves system using full pivoting 
%                    Gaussian elimination
% Input:
%     A: n x n matrix
%     b: n x 1 vector
% Output:
%     x: n x 1 vector
%     singular: scalar value indicating whether the matrix is singular
%               (0 = non-singular, 1 = singular)
%

dim = size(A,1);
x = [];
singular = 0; % assuming not singular

% check dimensions of input matrix and vector
if (size(A,2) ~= dim)
    error('A needs to be n x n.');
end
if (size(b,1) ~= dim || size(b,2) ~= 1)
    error('b needs to be n x 1');
end;

% full pivoting vectors for indexing into the matrix A
row = 1:dim;
col = 1:dim;

for k =1:dim-1

   % find max magnitude entry in remaining part of matrix
   [maxRow, maxCol] = largestMagnitudeLocation(A,k);
   if (abs(A(row(maxRow),col(maxCol))) < 10^-9)
       showMatrix(A);
       fprintf('Matrix is singular');
       singular = 1;
       return;
   end
   
   swapRows(k,maxRow);
   swapCols(k,maxCol);
   
   % do the Gaussian elimination
   [A, b] = rowReduce(A, b, k);
   
end

% check to see if the A(n,n) entry is zero in fully reduced row form 
if (abs(A(row(dim),col(dim))) < 10^-9) 
    showMatrix(A);
    fprintf('Matrix is singular');
    singular = 1;
    return;
end

% generate the solution to the system
x = backSub(A,b);

% end of the body of fpGelim() - the main full pivoting function

% --------------------------------------------------------------
%
%    Helper functions
%
%

    function showMatrix(A1)
    % displays the contents of a matrix according to
    % the full pivoting indexing into it
        
        dim1 = size(A1,1);
        
        for i=1:dim1
            for j = 1:dim1
                fprintf('%d',A1(row(i),col(j)));
                if j ~= dim1 
                    fprintf('  ');
                end
            end
            
            fprintf('\n');
        end
        
        fprintf('\n');
    end

    function [maxRow,maxCol] = largestMagnitudeLocation(A,k)
    % finds largest magnitude entry in submatrix of A
    % starting at A(k,k)
    
        max = abs(A(row(k),col(k)));
        maxRow = k;
        maxCol = k;
        
        for r2 = k:dim
            for c2 = k:dim
                if abs(A(row(r2),col(c2))) > max
                    max = abs(A(row(r2),col(c2)));
                    maxRow = r2;
                    maxCol = c2;
                end                
            end
        end        
    end

    function swapRows(r1,r2)
        
        temp = row(r1);
        row(r1) = row(r2);
        row(r2) = temp;
    end

    function swapCols(c1,c2)
        
        temp = col(c1);
        col(c1) = col(c2);
        col(c2) = temp;
    end


    function [A,b] = rowReduce(A, b, k)
    % does Gaussian elimination
    
        % multiplier from and for the kth row
        m1 = 1/A(row(k),col(k));

        % Creating vector based on row k that is almost in the form
        % that will be subtracted from the below rows. When subtraction
        % actually occurs, this vector will have needed to be temporarily
        % multiplied by the appropriate multiplier from the row being
        % reduced.
        sub = zeros(1, dim);
        for k1 = k+1:dim
           sub(1,k1) = A(row(k),col(k1))*m1;
        end
        bTemp = b(row(k),1) * m1;
        
        for r1 = k+1:dim
            m2 = A(row(r1),col(k));
            A(row(r1),col(k)) = 0;
            
            for c1 = k+1:dim
                A(row(r1),col(c1)) = A(row(r1),col(c1)) - sub(1,c1)*m2;
            end
            
            b(row(r1),1) = b(row(r1),1) - bTemp * m2;
        end
        
    end

    function x = backSub(A,b)
    % find x by doing back substitution on a triangular matrix
    
        n = size(A,1);
        x = zeros(n,1);
        
        for r1 = n:-1:1
        
            x(col(r1),1) = b(row(r1),1);
            
            for j = r1+1:n
                x(col(r1),1) = x(col(r1),1) - A(row(r1), col(j))*x(col(j),1);
            end
            
            x(col(r1),1) = x(col(r1),1) / A(row(r1),col(r1));

        end
                
    end

end  % end of fpGelim() - the main full pivoting function
